# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Get-IntelEthernet
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory = $false, Position = 0, ValueFromPipeline = $true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesGet = @()
        $script:WarningMessagesGet = @()
        $FinalObject = @()
        GetAllSupportedEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue
        }
        GetIntelDriverInfo
        $script:MSNetHwInfo = Get-NetAdapterHardwareInfo -ErrorAction SilentlyContinue
        $script:MSNetAdvProperty = Get-NetAdapterAdvancedProperty -ErrorAction SilentlyContinue
        $AdapterPropertiesNames = @("NetCfgInstanceId", "DriverVersion", "Port")
        $972Key = Get-ItemProperty -Path "HKLM:\SYSTEM\CurrentControlSet\Control\Class\{4d36e972-e325-11ce-bfc1-08002be10318}\*" -Name $AdapterPropertiesNames -ErrorAction SilentlyContinue
    }
    Process
    {
        $Adapters = $Adapter
        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesGet)
        $AdapterNamesArray = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesGet))

        foreach ($a in $AdapterNamesArray)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesGet += $TmpStatusMsg
            }

            $SupportedSpeeds = GetAdapterSupportedSpeeds $a
            $AdapterStatuses = GetAdapterStatus $a
            $BusTypes = GetAdapterBusType $a
            $Capabilities = GetAdapterCapabilities $a $SupportedSpeeds
            $ConnectionNames = GetAdapterConnectionName $a
            $DDPPackageNameAndVersions = GetAdapterDDPPackageNameAndVersion $a
            $DDPPackageTrackIds = GetAdapterDDPPackageTrackId $a
            $DeviceStatuses = GetAdapterDeviceStatus $a $SupportedSpeeds
            $DriverVersion = GetDriverVersion $a
            $EEEStatuses = GetAdapterEEEStatus $a
            $ETrackIDs = GetAdapterETrackId $a
            $FullDuplex = GetFullDuplex $a
            $LinkSpeedDuplex = GetLinkSpeedDuplex $a
            $MaxSpeeds = GetAdapterMaxSpeed $a $SupportedSpeeds
            $MediaTypes = GetAdapterMediaType $a
            $MiniPortNames = GetAdapterMiniPortName $a
            $NVMVersions = GetAdapterNVMVersion $a
            $NegotiatedLinkSpeed = GetAdapterNegotiatedLinkSpeed $a
            $NegotiatedLinkWidth = GetAdapterNegotiatedLinkWidth $a
            $NetlistVersions = GetAdapterNetlistVersion $a
            $OemFwVersions = GetOemFwVersion $a
            $OriginalDisplayNames = GetOriginalDisplayName $a
            $PCIDeviceIDs = GetAdapterPCIDeviceID $a
            $PartNumbers = GetAdapterPartNumber $a
            $PciLocations = GetAdapterPCILocation $a
            $RegistryValues = GetAdapterPropertiesFromRegistry $a $972Key
            $SanMacAddresses = GetAdapterSanMacAddress $a

            # Assemble it all together in PSCustomObject
            $FinalObject += [PsCustomObject] @{
                PSTypeName          = 'IntelEthernetAdapter'
                AdapterStatus       = $AdapterStatuses
                BusType             = $BusTypes.BusType
                BusTypeString       = $BusTypes.BusTypeString
                Capabilities        = $Capabilities
                ConnectionName      = $ConnectionNames
                DDPPackageName      = $DDPPackageNameAndVersions.Name
                DDPPackageTrackId   = $DDPPackageTrackIds
                DDPPackageVersion   = $DDPPackageNameAndVersions.Version
                DeviceStatus        = $DeviceStatuses.DeviceStatus
                DeviceStatusString  = $DeviceStatuses.DeviceStatusString
                DriverVersion       = $DriverVersion
                EEE                 = $EEEStatuses.EEEStatus
                EEEString           = $EEEStatuses.EEEStatusString
                ETrackID            = $ETrackIDs
                FWVersion           = $OemFwVersions
                FullDuplex          = $FullDuplex
                Location            = $PciLocations
                MaxSpeed            = $MaxSpeeds
                MediaType           = $MediaTypes.MediaType
                MediaTypeString     = $MediaTypes.MediaTypeString
                MiniPortInstance    = $MiniPortNames.Instance
                MiniPortName        = $MiniPortNames.Name
                NVMVersion          = $NVMVersions
                Name                = $a
                NegotiatedLinkSpeed       = $NegotiatedLinkSpeed.NegotiatedLinkSpeed
                NegotiatedLinkSpeedString = $NegotiatedLinkSpeed.NegotiatedLinkSpeedString
                NegotiatedLinkWidth       = $NegotiatedLinkWidth.NegotiatedLinkWidth
                NegotiatedLinkWidthString = $NegotiatedLinkWidth.NegotiatedLinkWidthString
                NetlistVersion      = $NetlistVersions
                OriginalDisplayName = $OriginalDisplayNames
                PCIDeviceID         = $PCIDeviceIDs
                PartNumber          = $PartNumbers
                PortNumber          = $RegistryValues.Port
                PortNumberString    = $RegistryValues.PortString
                SANMacAddress       = $SanMacAddresses
                Speed               = $LinkSpeedDuplex.Speed
                SpeedString         = $LinkSpeedDuplex.SpeedString
                }
        }
    }
    End
    {
        $FinalObject | Sort-Object -Property Location

        foreach ($WarningMessage in $script:WarningMessagesGet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesGet)
        {
            Write-Error $ErrorMessage
        }
    }
}


function GetAdapterBusType($AdapterName)
{
    $BusType       = 0
    $BusTypeString = $Messages.Unknown

    foreach ($Bus in $script:BusTypesArray)
    {
        if (($null -ne $Bus) -and ($Bus.InstanceName -eq $AdapterName))
        {
            $BusType       = $Bus.BusType
            $BusTypeString = $BusTypeMap[[int]$Bus.BusType]
            break
        }
    }
    return [PsCustomObject] @{
        BusType       = $BusType
        BusTypeString = $BusTypeString }
}

function GetAdapterConnectionName($AdapterName)
{
    return ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).InterfaceAlias
}

function GetAdapterDDPPackageNameAndVersion($AdapterName)
{
    $Name    = $Messages.NotSupported
    $Version = $Messages.NotSupported

    foreach ($DDPPkgName in $script:DDPPkgNamesArray)
    {
       if (($null -ne $DDPPkgName) -and ($DDPPkgName.InstanceName -eq $AdapterName))
       {
            $Name    = [System.Text.Encoding]::ASCII.GetString($DDPPkgName.Name)
            $Version = $DDPPkgName.Major.ToString() + "." + $DDPPkgName.Minor.ToString()
            break
       }
    }
    return [PsCustomObject] @{
        Name    = $Name
        Version = $Version }
}


function GetAdapterDDPPackageTrackId($AdapterName)
{
    $TrackId = $Messages.NotSupported
    $Params = @{Version = [uint32]1; Size = [uint32]12; Type = [uint32]1;}
    $Result = InvokeCimMethod "IntlLan_GetTrackId" $AdapterName "WmiGetTrackId" $Params
    if (($null -ne $Result) -and ($Result.ReturnValue -eq $true))
    {
        $TrackId = '0x{0:X}' -f $Result.Track_Id
    }
    return $TrackId
}


function GetAdapterDeviceStatus($AdapterName, $SupportedSpeeds)
{
    $DeviceStatus = 0
    $DeviceStatusString = $Messages.Unknown

    $AdapterNames = $script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})

    foreach($TmpAdapter in $AdapterNames)
    {
        if ("Up" -eq $TmpAdapter.Status)
        {
            $MaxSpeed = GetAdapterMaxSpeed $AdapterName $SupportedSpeeds
            $CurrentSpeed = (GetLinkSpeedDuplex $AdapterName).Speed
            if ($CurrentSpeed -lt $MaxSpeed)
            {
                $DeviceStatus = 4
                $DeviceStatusString = $Messages.LinkUpNotMax
            }
            else
            {
                $DeviceStatus = 1
                $DeviceStatusString = $Messages.LinkUp
            }
        }
        elseif ("Disconnected" -eq $TmpAdapter.Status)
        {
            $DeviceStatus = 2
            $DeviceStatusString = $Messages.LinkDown
        }
        elseif ("Disabled" -eq $TmpAdapter.Status)
        {
            $DeviceStatus = 0
            $DeviceStatusString = $Messages.Disabled
        }
        elseif (-not ($script:PnpDevice.Where({$_.Name -eq $TmpAdapter.InterfaceDescription})).Service)
        {
            $DeviceStatus = 4
            $DeviceStatusString = $Messages.NotPresent
        }
    }

    return [PsCustomObject] @{
        DeviceStatus       = $DeviceStatus
        DeviceStatusString = $DeviceStatusString}

}

function GetAdapterETrackId($AdapterName)
{
    $ETrackId = $Messages.NotSupported
    foreach ($ETrackId in $script:ETrackIdsArray)
    {
       if (($null -ne $ETrackId) -and ($ETrackId.InstanceName -eq $AdapterName))
       {
           $ETrackId = '0x{0:X}' -f $ETrackId.Id
           break
       }
    }
    return $ETrackId
}


function GetAdapterNVMVersion($AdapterName)
{
    $Version = $Messages.NotSupported
    foreach ($NVMVersion in $script:NVMVersionsArray)
    {
       if (($null -ne $NVMVersion) -and ($NVMVersion.InstanceName -eq $AdapterName))
       {
            $Version = (($NVMVersion.Version -band 0xffff) -shr 8).ToString() + "." + (($NVMVersion.Version -band 0xff)).ToString("X2")
            break
       }
    }
    return $Version
}

function GetAdapterNetlistVersion($AdapterName)
{
    $NetlistVersion = $Messages.NotSupported
    $Params = @{Version = [uint32]1; Size = [uint32]12; Type = [uint32]1;}
    $Result = InvokeCimMethod "IntlLan_GetNVMNetlistInfo" $AdapterName "WmiGetNVMNetlistInfo" $params
    if (($null -ne $Result) -and ($Result.ReturnValue -eq $true))
    {
        $NetlistVersion = [System.Text.Encoding]::ASCII.GetString($Result.VersionStr)
    }
    return $NetlistVersion
}

function GetAdapterPartNumber($AdapterName)
{
    $PartNumberString = $Messages.NotSupported
    foreach ($PartNumber in $script:PartNumbersArray)
    {
       if (($null -ne $PartNumber) -and ($PartNumber.InstanceName -eq $AdapterName))
       {
            $PartNumberString = [System.Text.Encoding]::ASCII.GetString($PartNumber.PartNumberString)
            break
       }
    }
    return $PartNumberString
}

function GetAdapterSanMacAddress($AdapterName)
{
    $MacAddress = $Messages.NotSupported

    foreach ($SanMacAddress in $script:SanMacAddressesArray)
    {
        if (($null -ne $SanMacAddress) -and ($SanMacAddress.InstanceName -eq $AdapterName))
        {
            $MacAddress = ""
            for ($i = 0; $i -lt 6; $i++)
            {
                # convert to string hex representation; X - hex, 2 - add leading zeroes if needed
                $MacAddress += $SanMacAddress.SanMacAddr[$i].ToString("X2")
            }
            break
        }
    }
    return $MacAddress
}

function GetAdapterMediaType($AdapterName)
{
    $MediaType = [PsCustomObject] @{
        MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_UNKNOWN
        MediaTypeString = $Messages.Unknown }

    $Service = ($script:SupportedAdapters.Where({$_.Name -eq $AdapterName})).Service

    switch ($Service)
    {
        {($_ -eq 'icea') -or ($_ -eq 'scea')}   { $MediaType = GetAdapterMediaTypeIce $AdapterName; break }
        {($_ -eq 'i40ea') -or ($_ -eq 'i40eb')} { $MediaType = GetAdapterMediaTypeI40e $AdapterName; break }
    }
    return $MediaType
}

function GetAdapterMediaTypeIce($AdapterName)
{
    foreach ($PhyInfo in $script:PhyInfoArray)
    {
        if (($null -ne $PhyInfo) -and ($PhyInfo.InstanceName -eq $AdapterName) -and ($PhyInfo.PhyInfo.Length -ge 8))
        {
            # Interpreting PhyInfo array values:
            # 0 == PHY type
            # 1 == link info <-- Bit 0 (value of 1 means has link)
            # 2 == an_info
            # 3 == ext_info
            # 4 == module type [0]
            # 5 == module type [1]
            # 6 == module type [2]
            # 7 == media interface  <-- 1=Backplane, 2=QSFP, 3=SFP
            $PhyType = $PhyInfo.PhyInfo[0];
            $LinkInfo = $PhyInfo.PhyInfo[1];
            $MediaInterface = $PhyInfo.PhyInfo[7];

            $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Unknown
            $MediaTypeString = $Messages.Unknown

            if ($LinkInfo -band 0x01)
            {
                switch ($PhyType)
                {
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GSFIAOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL25GAUIAOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL40GXLAUIAOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL50GLAUI2AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL50GAUI2AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL50GAUI1AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL100GCAUI4AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL100GAUI4AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL100GAUI2AOC_ACC}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_AOC_ACC
                        $MediaTypeString = $Messages.AOCACC
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL5GBaseKR,
                    [int][CVL_PHY_TYPE]::CVL10GBaseKR,
                    [int][CVL_PHY_TYPE]::CVL25GBaseKR}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR
                        $MediaTypeString = $Messages.BackplaneKR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseKR1)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR1
                        $MediaTypeString = $Messages.BackplaneKR1
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseKR2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR2
                        $MediaTypeString = $Messages.BackplaneKR2
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseKR2PAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR2_PAM4
                        $MediaTypeString = $Messages.BackplaneKR2PAM4
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseKR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseKR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR4
                        $MediaTypeString = $Messages.BackplaneKR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseKR4PAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR4_PAM4
                        $MediaTypeString = $Messages.BackplaneKR4PAM4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseKRPAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR_PAM4
                        $MediaTypeString = $Messages.BackplaneKRPAM4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseKRS)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR_S
                        $MediaTypeString = $Messages.BackplaneKRS
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL1000BaseKX,
                    [int][CVL_PHY_TYPE]::CVL2Point5GBaseKX}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KX
                        $MediaTypeString = $Messages.BackplaneKX
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GSFIC2C,
                    [int][CVL_PHY_TYPE]::CVL25GAUIC2C,
                    [int][CVL_PHY_TYPE]::CVL40GXLAUI,
                    [int][CVL_PHY_TYPE]::CVL50GLAUI2,
                    [int][CVL_PHY_TYPE]::CVL50GAUI2,
                    [int][CVL_PHY_TYPE]::CVL50GAUI1,
                    [int][CVL_PHY_TYPE]::CVL100GCAUI4,
                    [int][CVL_PHY_TYPE]::CVL100GAUI4,
                    [int][CVL_PHY_TYPE]::CVL100GAUI2}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Chip_to_Chip
                        $MediaTypeString = $Messages.ChiptoChip
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL1000BaseT,
                    [int][CVL_PHY_TYPE]::CVL2Point5GBaseT,
                    [int][CVL_PHY_TYPE]::CVL5GBaseT,
                    [int][CVL_PHY_TYPE]::CVL10GBaseT,
                    [int][CVL_PHY_TYPE]::CVL25GBaseT}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Copper_T
                        $MediaTypeString = $Messages.CopperT
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100BaseTX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Copper_TX
                        $MediaTypeString = $Messages.CopperTX
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL10GSFIDA)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach
                        $MediaTypeString = $Messages.DirectAttach
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseCP2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CP2
                        $MediaTypeString = $Messages.DirectAttachCP2
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseCR)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR
                        $MediaTypeString = $Messages.DirectAttachCR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseCR1)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR1
                        $MediaTypeString = $Messages.DirectAttachCR1
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseCR2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR2
                        $MediaTypeString = $Messages.DirectAttachCR2
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseCR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseCR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR4
                        $MediaTypeString = $Messages.DirectAttachCR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseCRPAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR_PAM4
                        $MediaTypeString = $Messages.DirectAttachCRPAM4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseCRS)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR_S
                        $MediaTypeString = $Messages.DirectAttachCRS
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseDR)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_DR
                        $MediaTypeString = $Messages.FiberDR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseFR)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_FR
                        $MediaTypeString = $Messages.FiberFR
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GBaseLR,
                    [int][CVL_PHY_TYPE]::CVL25GBaseLR,
                    [int][CVL_PHY_TYPE]::CVL50GBaseLR}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_LR
                        $MediaTypeString = $Messages.FiberLR
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseLR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseLR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_LR4
                        $MediaTypeString = $Messages.FiberLR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL1000BaseLX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_LX
                        $MediaTypeString = $Messages.FiberLX
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GBaseSR,
                    [int][CVL_PHY_TYPE]::CVL25GBaseSR,
                    [int][CVL_PHY_TYPE]::CVL50GBaseSR}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SR
                        $MediaTypeString = $Messages.FiberSR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseSR2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SR2
                        $MediaTypeString = $Messages.FiberSR2
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseSR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseSR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SR4
                        $MediaTypeString = $Messages.FiberSR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL1000BaseSX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SX
                        $MediaTypeString = $Messages.FiberSX
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL2Point5gBaseX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_X
                        $MediaTypeString = $Messages.FiberX
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL100MSGMII,
                    [int][CVL_PHY_TYPE]::CVL1GSGMII}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_SGMII
                        $MediaTypeString = $Messages.SGMII
                        break
                    }
                    default
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Unknown
                        $MediaTypeString = $Messages.Unknown
                        break
                    }
                }
            }
            else
            {
                switch ($MediaInterface)
                {
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_BACKPLANE)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_BACKPLANE
                        $MediaTypeString = $Messages.Backplane
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_QSFP)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_QSFP
                        $MediaTypeString = $Messages.QSFP
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_SFP)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_SFP
                        $MediaTypeString = $Messages.SFP
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_BASE_T)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_BASE_T
                        $MediaTypeString = $Messages.CopperT
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_SGMII)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_SGMII
                        $MediaTypeString = $Messages.SGMII
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_FIBER)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_FIBER
                        $MediaTypeString = $Messages.Fiber
                        break
                    }
                    default
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Unknown
                        $MediaTypeString = $Messages.Unknown
                        break
                    }
                }
            }
       }
    }

    return [PsCustomObject] @{
        MediaType = $MediaType
        MediaTypeString = $MediaTypeString }
}

function GetAdapterMediaTypeI40e($AdapterName)
{
    foreach ($PhyInfo in $script:PhyInfoArray)
    {
        if (($null -ne $PhyInfo) -and ($PhyInfo.InstanceName -eq $AdapterName) -and ($PhyInfo.PhyInfo.Length -ge 13))
        {
            $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_UNKNOWN
            $MediaTypeString = $Messages.Unknown

            $PhyType = $PhyInfo.PhyInfo[0];
            $LinkType = ([uint32]$PhyInfo.PhyInfo[11] -shl 24) + ([uint32]$PhyInfo.PhyInfo[10] -shl 16) + ([uint32]$PhyInfo.PhyInfo[9] -shl 8) + [uint32]$PhyInfo.PhyInfo[8];
            $LinkTypeExt = $PhyInfo.PhyInfo[12];

            if ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_EMPTY)
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_UNKNOWN
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_KR4) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_KR) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_KR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_BACKPLANE
                $MediaTypeString = $Messages.Backplane
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_SFP_Cu) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_CR1))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFPDIRECTATTACH
                $MediaTypeString = $Messages.SFPDirectAttach
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_LR4) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_LR) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_LR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_LR_FIBER
                $MediaTypeString = $Messages.LRFiber
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_SR) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_SR4) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_SR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SR_FIBER
                $MediaTypeString = $Messages.SRFiber
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_T) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_1000BASE_T) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_100BASE_TX) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_2_5GBASE_T) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_5GBASE_T))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_COPPER
                $MediaTypeString = $Messages.Copper
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_CR1_CU) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_CR4_CU) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_CR4))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_QSFPDIRECTATTACH
                $MediaTypeString = $Messages.QSFPDirectAttach
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_KX4) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_1000BASE_KX))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_KX4BACKPLANE
                $MediaTypeString = $Messages.KXBackplane
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_XAUI) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_XLAUI))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_XAUI
                $MediaTypeString = $Messages.XAUI
            }
            elseif (($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25G_AOC) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25G_ACC) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_CR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFP28DIRECTATTACH
                $MediaTypeString = $Messages.SFP28DirectAttach
            }
            # old FW or HW different than XL710
            elseif ($LinkTypeExt -eq 0 -and $LinkType -eq 0)
            {
                if (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_KR4) -or
                    ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_KR) -or
                    ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_KR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_BACKPLANE
                    $MediaTypeString = $Messages.Backplane
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_SFPP_CU) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_CR1))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFPDIRECTATTACH
                    $MediaTypeString = $Messages.SFPDirectAttach
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_LR4) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_LR) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_LR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_LR_FIBER
                    $MediaTypeString = $Messages.LRFiber
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_SR) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_SR4) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_SR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SR_FIBER
                    $MediaTypeString = $Messages.SRFiber
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_T) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_1000BASE_T) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_100BASE_TX))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_COPPER
                    $MediaTypeString = $Messages.Copper
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_CR1_CU) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_CR4_CU) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_CR4))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_QSFPDIRECTATTACH
                    $MediaTypeString = $Messages.QSFPDirectAttach
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_KX4) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_1000BASE_KX))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_KX4BACKPLANE
                    $MediaTypeString = $Messages.KXBackplane
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_XAUI) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_XLAUI))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_XAUI
                    $MediaTypeString = $Messages.XAUI
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_AOC) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_ACC) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_CR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFP28DIRECTATTACH
                    $MediaTypeString = $Messages.SFP28DirectAttach
                }
            }
        }
    }

    return [PsCustomObject] @{
        MediaType = $MediaType
        MediaTypeString = $MediaTypeString }
}

function GetAdapterPCILocation($AdapterName)
{
    $PCILocation = $Messages.NotSupported
    foreach ($HwInfo in $script:MSNetHwInfo)
    {
        if ($AdapterName -eq $HwInfo.ifDesc)
        {
            $PCILocation = $HwInfo.Bus.ToString() + ":" + $HwInfo.Device.ToString()  + ":" + $HwInfo.Function.ToString() + ":" + $HwInfo.Segment.ToString()
            break
        }
    }
    return $PCILocation
}

function GetOriginalDisplayName($AdapterName)
{
    return ($script:MSNetHwInfo.Where({$_.ifDesc -eq $AdapterName})).ifDesc
}

function GetOemFwVersion($AdapterName)
{
    $Version = $Messages.NotSupported
    foreach ($FwVersion in $script:FwVersionsArray)
    {
        if ($FwVersion.InstanceName -eq $AdapterName)
        {
            # driver can return and array of zeroes - don't attempt to construct a string using it
            if ($FwVersion.SingleNvmVersion[0] -ne 0)
            {
                $Version = [System.Text.Encoding]::ASCII.GetString($FwVersion.SingleNvmVersion)
            }
            break
        }
    }
    return $Version
}

function GetAdapterPCIDeviceID($AdapterName)
{
    return ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).PnPDeviceID
}

function GetAdapterNegotiatedLinkWidth($AdapterName)
{
    $NegotiatedLinkWidth = 0
    $NegotiatedLinkWidthString = $Messages.NotAvailable

    $AdapterHWInfo = $script:MSNetHwInfo.Where({$_.ifDesc -eq $AdapterName})

    if ($AdapterHWInfo)
    {
        $NegotiatedLinkWidth = $AdapterHWInfo.PciExpressCurrentLinkWidth
        $NegotiatedLinkWidthString = "x" + $NegotiatedLinkWidth.ToString()
    }

    return [PsCustomObject] @{
        NegotiatedLinkWidth = $NegotiatedLinkWidth
        NegotiatedLinkWidthString = $NegotiatedLinkWidthString }
}

function GetAdapterNegotiatedLinkSpeed($AdapterName)
{
    $NegotiatedLinkSpeed = ($script:MSNetHwInfo.Where({$_.ifDesc -eq $AdapterName})).PciExpressCurrentLinkSpeedEncoded
    switch ($NegotiatedLinkSpeed)
    {
        0 {$NegotiatedLinkSpeedString = $Messages.Unknown; break}
        1 {$NegotiatedLinkSpeedString = $Messages.NegLaneSpeed25; break}
        2 {$NegotiatedLinkSpeedString = $Messages.NegLaneSpeed50; break}
        3 {$NegotiatedLinkSpeedString = $Messages.NegLaneSpeed80; break}
        default {$NegotiatedLinkSpeedString = $Messages.Unknown; break}
    }
    return [PsCustomObject] @{
        NegotiatedLinkSpeed = $NegotiatedLinkSpeed
        NegotiatedLinkSpeedString = $NegotiatedLinkSpeedString }
}

function GetLinkSpeedDuplex($AdapterName)
{
    $AdapterObj = $script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})
    $Speed = $AdapterObj.Speed
    $SpeedString = $AdapterObj.LinkSpeed
    $FullDuplex = $AdapterObj.FullDuplex

    if ("Up" -ne $AdapterObj.Status)
    {
        $Speed = 0
        $SpeedString = $Messages.NotAvailable
    }
    elseif ($true -eq $FullDuplex)
    {
        $SpeedString += " " + $Messages.FullDuplex
    }

    return [PsCustomObject] @{
        Speed = $Speed
        SpeedString = $SpeedString }
}

function GetFullDuplex($AdapterName)
{
    $FullDuplex = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).FullDuplex
    if (-not $FullDuplex)
    {
        $FullDuplex = ""
    }
    return $FullDuplex
}

function GetAdapterPropertiesFromRegistry($AdapterName, $972Key)
{
    # Individual Adapter GUID
    $AdapterInstanceID = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).InterfaceGuid
    $AdapterRegKey = $972Key.Where({$_.NetCfgInstanceId -eq $AdapterInstanceID})

    switch ($AdapterRegKey.Port)
    {
        0 {$PortNumberString = $Messages.PortA; break}
        1 {$PortNumberString = $Messages.PortB; break}
        2 {$PortNumberString = $Messages.PortC; break}
        3 {$PortNumberString = $Messages.PortD; break}
        default {$PortNumberString = $Messages.NotSupported; break}
    }

    return [PsCustomObject] @{
        Port       = $AdapterRegKey.Port
        PortString = $PortNumberString }
}

function GetDriverVersion($AdapterName)
{
    $DriverVersion = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).DriverVersion
    return $DriverVersion
}

function GetAdapterMiniPortName($AdapterName)
{
    $Name = ($script:PnpDevice.Where({$_.Name -eq $AdapterName})).Service
    $Instance = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).InterfaceGuid
    return [PsCustomObject] @{
        Name     = $Name
        Instance = $Instance }
}

function GetAdapterMaxSpeed($AdapterName, $SupportedSpeeds)
{
    if ($SupportedSpeeds.Length -gt 0)
    {
        # array is sorted, so we just return the last element
        return $SupportedSpeeds[-1]
    }
    return 0
}

function GetAdapterSupportedSpeeds($AdapterName)
{
    $SpeedDuplex = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -eq "*SpeedDuplex"})
    if ($SpeedDuplex)
    {
        $RegistryValues = ($SpeedDuplex).ValidRegistryValues
    }

    $SupportedSpeeds = @()

    foreach ($i in $RegistryValues)
    {
        $SupportedSpeeds += $SupportedSpeedsMap[$i]
    }

    return $SupportedSpeeds | Sort-Object
}


function GetAdapterEEEStatus($AdapterName)
{
    $EEEStatus = 0
    $EEEStatusString = $Messages.NotSupported

    $Service = ($script:SupportedAdapters.Where({$_.Name -eq $AdapterName})).Service


    foreach ($EEE in $script:EEELinkStatusArray)
    {
        if (($null -ne $EEE) -and ($EEE.InstanceName -eq $AdapterName))
        {
            if ($EEE.EEELinkStatus -eq $false)
            {
                $EEELinkAdvertisement = 0
                if ($Service -in @("i40ea", "i40eb"))
                {
                    $EEELinkAdvertisement = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -eq "EEELinkAdvertisement"})
                }

                if ($EEELinkAdvertisement -gt 0)
                {
                    $EEEStatus = 3 #Not Negotiated
                    $EEEStatusString = $Messages.NotNegotiated
                }
                else
                {
                    $EEEStatus = 1 #Disabled
                    $EEEStatusString = $Messages.Disabled
                }
            }
            else
            {
                $EEEStatus = 2 #Active
                $EEEStatusString = $Messages.Active
            }
        }
    }

    return [PsCustomObject] @{
        EEEStatus       = $EEEStatus
        EEEStatusString = $EEEStatusString }
}

function GetAdapterStatus($AdapterName)
{
    $AdapterStatus = [ADAPTER_STATUS]::Installed -bor [ADAPTER_STATUS]::DriverLoaded -bor [ADAPTER_STATUS]::HasDiag

    $LinkStatus = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).Status
    if ($LinkStatus -eq 'Up')
    {
        $AdapterStatus = $AdapterStatus -bor [ADAPTER_STATUS]::HasLink
    }
    return $AdapterStatus
}

function GetAdapterCapabilities($AdapterName, $SupportedSpeeds)
{
    $Capabilities = @([int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_VENDOR_INTEL)

    foreach ($SupportedSpeed in $SupportedSpeeds)
    {
        switch ($SupportedSpeed)
        {
            10000000     {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_10_MBPS; break}
            100000000    {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_100_MBPS; break}
            1000000000   {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_1000_MBPS; break}
            2500000000   {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_2500_MBPS; break}
            5000000000   {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_5000_MBPS; break}
            10000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_10000_MBPS; break}
            40000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_40000_MBPS; break}
            25000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_25000_MBPS; break}
            50000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_50000_MBPS; break}
            100000000000 {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_100000_MBPS; break}
        }
    }
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_PERFORMANCE_PROFILE
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_DIAGNOSTIC_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_FLASH_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_CYPRESS
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_IDENTIFY_ADAPTER_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_NDIS_IOAT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_EXTENDED_DMIX_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_DCB
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_JUMBO_FRAMES

    return , ($Capabilities | Sort-Object)
}

function GetIntelDriverInfo()
{
    $script:BusTypesArray        = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_BusType -Property BusType -ErrorAction SilentlyContinue
    $script:DDPPkgNamesArray     = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetPackageInfo -Property Name, Major, Minor -ErrorAction SilentlyContinue
    $script:ETrackIdsArray       = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_EetrackId -Property Id -ErrorAction SilentlyContinue
    $script:NVMVersionsArray     = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_EepromVersion -Property Version -ErrorAction SilentlyContinue
    $script:SanMacAddressesArray = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetSanMacAddress -Property SanMacAddr -ErrorAction SilentlyContinue
    $script:PartNumbersArray     = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_PartNumberString -Property PartNumberString -ErrorAction SilentlyContinue
    $script:PhyInfoArray         = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetPhyInfo -Property PhyInfo -ErrorAction SilentlyContinue
    $script:FwVersionsArray      = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetOemProductVer -Property SingleNvmVersion -ErrorAction SilentlyContinue
    $script:EEELinkStatusArray   = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_EEELinkStatus -Property EEELinkStatus -ErrorAction SilentlyContinue
}

# SIG # Begin signature block
# MIIorAYJKoZIhvcNAQcCoIIonTCCKJkCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCABPeuN9xOydxAb
# y0kPWyBly/c+bT5B9AtpIe3WVgI6T6CCEgUwggWeMIIEhqADAgECAhEAzS1l4rws
# CIvYBjRVawV4ujANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJHQjEbMBkGA1UE
# CBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQK
# Ew9TZWN0aWdvIExpbWl0ZWQxJDAiBgNVBAMTG1NlY3RpZ28gUlNBIENvZGUgU2ln
# bmluZyBDQTAeFw0yMTA0MDIwMDAwMDBaFw0yMzA0MDIyMzU5NTlaMIGEMQswCQYD
# VQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEUMBIGA1UEBwwLU2FudGEgQ2xh
# cmExGjAYBgNVBAoMEUludGVsIENvcnBvcmF0aW9uMRIwEAYDVQQLDAlTb2xhcktN
# Q1MxGjAYBgNVBAMMEUludGVsIENvcnBvcmF0aW9uMIIBojANBgkqhkiG9w0BAQEF
# AAOCAY8AMIIBigKCAYEA7CCN9iKpDHOrRceKhlXFP/tf6Lllw2H2fR9KVI4/fQIx
# MU1hXwnlHmAzMCY7IgcCFY4p3F5/MJGKaqYngwOo28Zo6Q1N6ukysA7PSavmF2RY
# WD6VFeya/2H0PoNeRFjHaRzSeynFFeJAFew9r7UReUwM/507sxZYPQuWWIdAEK7H
# Dqp2VlHmgZOXVGHhNO6GFOKpC/C01g6X3x6OquddRNMt5UrZzZzDo5MpJz9SBB2V
# jiqwZ80dvNR2W2xi90cIHh4BkXvB54UNkp4VTVu16T0k3cweo+C39U7GrCAr5Axz
# DETjBvhNtP1sf9SoRV7xY6g5wssfI7yYT9J0gsifn/Vy8MWH355TPoA+PVhbAu0m
# 9FMz4EWu55nnUurNML2jaUxsos21/7ELat12kWC0tq9fhkODjKO8X9PuiBHflZLk
# d3F4QcSMvuGocWGqE77VV3vn8jlvigm2TOV0CfGTQajGMX0jeTRZ19fzBNkt2X9d
# SSGolI/Kj1gSvCggpkUBAgMBAAGjggGQMIIBjDAfBgNVHSMEGDAWgBQO4TqoUzox
# 1Yq+wbutZxoDha00DjAdBgNVHQ4EFgQUshkNuM2SdwJnW4vFy8c4FtUTrbQwDgYD
# VR0PAQH/BAQDAgeAMAwGA1UdEwEB/wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# EQYJYIZIAYb4QgEBBAQDAgQQMEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMCMCUw
# IwYIKwYBBQUHAgEWF2h0dHBzOi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEATBD
# BgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29S
# U0FDb2RlU2lnbmluZ0NBLmNybDBzBggrBgEFBQcBAQRnMGUwPgYIKwYBBQUHMAKG
# Mmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1JTQUNvZGVTaWduaW5nQ0Eu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQsFAAOCAQEAVadLNRW4f/pKMqrbn0BdOoQ8/1EJ87gvVfosei2bLwTEvpmv
# mn2n561H6AFedtIJ6L4FmXII4M4r20i+5LREbI6PpKDmOAf4xW7POxfCRvkTQAZO
# 3zoVxjMQBXo7cZVF1xHCdviXzD1usuIiCF8DLm6z4O/kyeFFNcn816yPQct91Pnk
# SBBVvL+Kwu8xvR+ZIQy632WUA4HnNpRdFnVSzUifEg2GrtsKZR8k+rm2o8K8yjJq
# 3SznwgJQCMVMh3CtRtUwE/c7o/6rvm53fTYJDd3aoPHVgH6S2WqS3+3mQG7A6hTD
# nrP/mYnS4PF7XzxxjZhUlhy4G/MarJPvT9IrNDCCBfUwggPdoAMCAQICEB2iSDBv
# myYY0ILgln0z02owDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMV
# VGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENl
# cnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE4MTEwMjAwMDAwMFoXDTMwMTIzMTIz
# NTk1OVowfDELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3Rl
# cjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSQw
# IgYDVQQDExtTZWN0aWdvIFJTQSBDb2RlIFNpZ25pbmcgQ0EwggEiMA0GCSqGSIb3
# DQEBAQUAA4IBDwAwggEKAoIBAQCGIo0yhXoYn0nwli9jCB4t3HyfFM/jJrYlZilA
# hlRGdDFixRDtsocnppnLlTDAVvWkdcapDlBipVGREGrgS2Ku/fD4GKyn/+4uMyD6
# DBmJqGx7rQDDYaHcaWVtH24nlteXUYam9CflfGqLlR5bYNV+1xaSnAAvaPeX7Wpy
# vjg7Y96Pv25MQV0SIAhZ6DnNj9LWzwa0VwW2TqE+V2sfmLzEYtYbC43HZhtKn52B
# xHJAteJf7wtF/6POF6YtVbC3sLxUap28jVZTxvC6eVBJLPcDuf4vZTXyIuosB69G
# 2flGHNyMfHEo8/6nxhTdVZFuihEN3wYklX0Pp6F8OtqGNWHTAgMBAAGjggFkMIIB
# YDAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQUDuE6
# qFM6MdWKvsG7rWcaA4WtNA4wDgYDVR0PAQH/BAQDAgGGMBIGA1UdEwEB/wQIMAYB
# Af8CAQAwHQYDVR0lBBYwFAYIKwYBBQUHAwMGCCsGAQUFBwMIMBEGA1UdIAQKMAgw
# BgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5j
# b20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwdgYIKwYB
# BQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnVzZXJ0cnVzdC5jb20v
# VVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQwJQYIKwYBBQUHMAGGGWh0dHA6Ly9v
# Y3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIBAE1jUO1HNEphpNve
# aiqMm/EAAB4dYns61zLC9rPgY7P7YQCImhttEAcET7646ol4IusPRuzzRl5ARokS
# 9At3WpwqQTr81vTr5/cVlTPDoYMot94v5JT3hTODLUpASL+awk9KsY8k9LOBN9O3
# ZLCmI2pZaFJCX/8E6+F0ZXkI9amT3mtxQJmWunjxucjiwwgWsatjWsgVgG10Xkp1
# fqW4w2y1z99KeYdcx0BNYzX2MNPPtQoOCwR/oEuuu6Ol0IQAkz5TXTSlADVpbL6f
# ICUQDRn7UJBhvjmPeo5N9p8OHv4HURJmgyYZSJXOSsnBf/M6BZv5b9+If8AjntIe
# Q3pFMcGcTanwWbJZGehqjSkEAnd8S0vNcL46slVaeD68u28DECV3FTSK+TbMQ5Lk
# uk/xYpMoJVcp+1EZx6ElQGqEV8aynbG8HArafGd+fS7pKEwYfsR7MUFxmksp7As9
# V1DSyt39ngVR5UR43QHesXWYDVQk/fBO4+L4g71yuss9Ou7wXheSaG3IYfmm8SoK
# C6W59J7umDIFhZ7r+YMp08Ysfb06dy6LN0KgaoLtO0qqlBCk4Q34F8W2WnkzGJLj
# tXX4oemOCiUe5B7xn1qHI/+fpFGe+zmAEc3btcSnqIBv5VPU4OOiwtJbGvoyJi1q
# V3AcPKRYLqPzW0sH3DJZ84enGm1YMIIGZjCCBE6gAwIBAgITMwAAAES3P/zvWs+i
# egAAAAAARDANBgkqhkiG9w0BAQUFADB/MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSkwJwYDVQQDEyBNaWNyb3NvZnQgQ29kZSBWZXJpZmljYXRp
# b24gUm9vdDAeFw0xNTA3MjIyMTAzNDlaFw0yNTA3MjIyMTAzNDlaMIGIMQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENp
# dHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNF
# UlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAIASZRc2DsPbCLPQrFcNdu3NJ9NMrVCDYeKqIE0J
# LWQJ3M6Jn8w9qez2z8Hc8dOx1ns3KBErR9o5xrw6GbRfpr19naNjQrZ28qk7K5H4
# 4m/Q7BYgkAk+4uh0yRi0kdRiZNt/owbxiBhqkCI8vP4T8IcUe/bkH47U5FHGEWdG
# CFHLhhRUP7wz/n5snP8WnRi9UY41pqdmyHJn2yFmsdSbeAPAUDrozPDcvJ5M/q8F
# ljUfV1q3/875PbcstvZU3cjnEjpNrkyKt1yatLcgPcp/IjSufjtoZgFE5wFORlOb
# M2D3lL5TN5BzQ/Myw1Pv26r+dE5px2uMYJPexMcM3+EyrsyTO1F4lWeL7j1W/gzQ
# aQ8bD/MlJmszbfduR/pzQ+V+DqVmsSl8MoRjVYnEDcGTVDAZE6zTfTen6106bDVc
# 20HXEtqpSQvf2ICKCZNijrVmzyWIzYS4sT+kOQ/ZAp7rEkyVfPNrBaleFoPMuGfi
# 6BOdzFuC00yz7Vv/3uVzrCM7LQC/NVV0CUnYSVgaf5I25lGSDvMmfRxNF7zJ7EMm
# 0L9BX0CpRET0medXh55QH1dUqD79dGMvsVBlCeZYQi5DGky08CVHWfoEHpPUJkZK
# UIGy3r54t/xnFeHJV4QeD2PW6WK61l9VLupcxigIBCU5uA4rqfJMlxwHPw1S9e3v
# L4IPAgMBAAGjgdAwgc0wEwYDVR0lBAwwCgYIKwYBBQUHAwMwEgYDVR0TAQH/BAgw
# BgEB/wIBAjAdBgNVHQ4EFgQUU3m/WqorSs9UgOHYm8Cd8rIDZsswCwYDVR0PBAQD
# AgGGMB8GA1UdIwQYMBaAFGL7CiFbf0NuEdoJVFBr9dKWcfGeMFUGA1UdHwROMEww
# SqBIoEaGRGh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3Rz
# L01pY3Jvc29mdENvZGVWZXJpZlJvb3QuY3JsMA0GCSqGSIb3DQEBBQUAA4ICAQBr
# IpM8PTlUcWRrDvLkPDARxSBKS4YPkvH/M3k62eSYpw5AoCKAfmGy4KcZzyaVMSpl
# 1GpPMYbqwMYuxWSMPUhZzQsvdD2UJhMQQtSXmCdePHbSeGkdGmTnBXJ14OtmQEOf
# jwxG/5dgpshnrRAIm2Km6b46itMHTZ9ykyW8BhHgLJA4Pmcc/RnXnpDOPcLg52Gs
# wOUE9R6ZVAyRDQFWcTeuJ9SeQyKlySfNTeVxEjkkpUFWh/+8VRQPJcqJ7seX5dIT
# /z1+GqCPP8gs16Nw0MdgwPzYPlHnl8Y+O+3PeL6KyuPE8qen7Z6uCAKPoFLbch7V
# O8NNn476m3DH+OO/bD+Sm+Q3PuxqjCn5waK/iz4aaWb7HGNPJgHJAsQ+0v/DQ6gb
# /Zn61LylueKTLzsBxdH0Oi9ow+Bkt1qVXkbMB4NpuzwFklZzNXNFmE582BKlt0Lp
# omP2QmAYcNE7bzHAh8fmceHzRhbp9bhys+ltH2ImSaNJi91ox4toVvfe/PqHJLgD
# gReP5fFnah2u03T3jKVdswuOQimWzknEd35mfAEXGmwUJMOwF3cF2BpAt4Zr2OR7
# QKx+305vJPkggIKMM+fl+inYndqLcF0ryR2CTAtny4RBnucGfhGDRC2KGe70f5rd
# eRw3GR6fP4wpug1cEIY3bEjNRV3NcLy80U1d2MW4djGCFf0wghX5AgEBMIGRMHwx
# CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNV
# BAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEkMCIGA1UEAxMb
# U2VjdGlnbyBSU0EgQ29kZSBTaWduaW5nIENBAhEAzS1l4rwsCIvYBjRVawV4ujAN
# BglghkgBZQMEAgEFAKBqMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisG
# AQQBgjcCAQsxDjAMBgorBgEEAYI3AgEWMC8GCSqGSIb3DQEJBDEiBCAtPyVhOZDg
# nRLUIumL15jztkT4/N6G7lOYe12stbj2ujANBgkqhkiG9w0BAQEFAASCAYAVyQHL
# EaKDfZmDjdoVfdAC/N/CCWixJagG83BZ4ouXLzXUJp1BY7DdRLzQZ8m29jJzb4ZK
# 5M82UyzO7rWYSCLPLoKB/VxqhmndknJMoI1w/XQTmZzN4xlDQng1PAiIFVUXTGx9
# QcwYha+DtX3IlEticvBnwn/k162NNxHyJTtzfcL5kM9gYilAhM95XLObRnusLPae
# LLTaImHsOipYthCJBzxDqlRoGmYDUcS2jyxnTcTBDrYlaMDG8xgv4DXM0eenASWG
# wSDAdnpOu5r4yZXj3XAmDfpX93YSHr/z4K2/nTpqHc9GIZV2tbIQAhThJw7pt6PS
# FiT5G05vkgEqM2bEhX6jGpXF3BAvtuvLGt+gtv2cA328gkVSQ9PlZNC+Rk8NcTml
# ZfIgJiZ3IaQWT6okPNLMD7+f7JYE74EIAz2LA86RxPomg5fBlloeoikbf+xS4K/u
# JWc+y7eM9Jsal9nhjYuLteI6/W9w9zUMs6mthvP6dGzxXftJEhQhaj4RKo2hghNQ
# MIITTAYKKwYBBAGCNwMDATGCEzwwghM4BgkqhkiG9w0BBwKgghMpMIITJQIBAzEP
# MA0GCWCGSAFlAwQCAgUAMIHvBgsqhkiG9w0BCRABBKCB3wSB3DCB2QIBAQYKKwYB
# BAGyMQIBATAxMA0GCWCGSAFlAwQCAQUABCCKrg64QsIMGIzCXGV0EaPpwQJjFdUX
# WuwHnsJADgYDygIUXMRdcydIPb0zDsZ0AZ4txG/XLCUYDzIwMjMwMTEwMjMwMDEw
# WqBupGwwajELMAkGA1UEBhMCR0IxEzARBgNVBAgTCk1hbmNoZXN0ZXIxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEsMCoGA1UEAwwjU2VjdGlnbyBSU0EgVGltZSBT
# dGFtcGluZyBTaWduZXIgIzOggg3qMIIG9jCCBN6gAwIBAgIRAJA5f5rSSjoT8r2R
# Xwg4qUMwDQYJKoZIhvcNAQEMBQAwfTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdy
# ZWF0ZXIgTWFuY2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2Vj
# dGlnbyBMaW1pdGVkMSUwIwYDVQQDExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5n
# IENBMB4XDTIyMDUxMTAwMDAwMFoXDTMzMDgxMDIzNTk1OVowajELMAkGA1UEBhMC
# R0IxEzARBgNVBAgTCk1hbmNoZXN0ZXIxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRl
# ZDEsMCoGA1UEAwwjU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBTaWduZXIgIzMw
# ggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQCQsnE/eeHUuYoXzMOXwpCU
# cu1aOm8BQ39zWiifJHygNUAG+pSvCqGDthPkSxUGXmqKIDRxe7slrT9bCqQfL2x9
# LmFR0IxZNz6mXfEeXYC22B9g480Saogfxv4Yy5NDVnrHzgPWAGQoViKxSxnS8JbJ
# RB85XZywlu1aSY1+cuRDa3/JoD9sSq3VAE+9CriDxb2YLAd2AXBF3sPwQmnq/ybM
# A0QfFijhanS2nEX6tjrOlNEfvYxlqv38wzzoDZw4ZtX8fR6bWYyRWkJXVVAWDUt0
# cu6gKjH8JgI0+WQbWf3jOtTouEEpdAE/DeATdysRPPs9zdDn4ZdbVfcqA23VzWLa
# zpwe/OpwfeZ9S2jOWilh06BcJbOlJ2ijWP31LWvKX2THaygM2qx4Qd6S7w/F7Kvf
# LW8aVFFsM7ONWWDn3+gXIqN5QWLP/Hvzktqu4DxPD1rMbt8fvCKvtzgQmjSnC//+
# HV6k8+4WOCs/rHaUQZ1kHfqA/QDh/vg61MNeu2lNcpnl8TItUfphrU3qJo5t/KlI
# mD7yRg1psbdu9AXbQQXGGMBQ5Pit/qxjYUeRvEa1RlNsxfThhieThDlsdeAdDHpZ
# iy7L9GQsQkf0VFiFN+XHaafSJYuWv8at4L2xN/cf30J7qusc6es9Wt340pDVSZo6
# HYMaV38cAcLOHH3M+5YVxQIDAQABo4IBgjCCAX4wHwYDVR0jBBgwFoAUGqH4YRkg
# D8NBd0UojtE1XwYSBFUwHQYDVR0OBBYEFCUuaDxrmiskFKkfot8mOs8UpvHgMA4G
# A1UdDwEB/wQEAwIGwDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUF
# BwMIMEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMIMCUwIwYIKwYBBQUHAgEWF2h0
# dHBzOi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEAjBEBgNVHR8EPTA7MDmgN6A1
# hjNodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29SU0FUaW1lU3RhbXBpbmdD
# QS5jcmwwdAYIKwYBBQUHAQEEaDBmMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnNl
# Y3RpZ28uY29tL1NlY3RpZ29SU0FUaW1lU3RhbXBpbmdDQS5jcnQwIwYIKwYBBQUH
# MAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBz
# 2u1ocsvCuUChMbu0A6MtFHsk57RbFX2o6f2t0ZINfD02oGnZ85ow2qxp1nRXJD9+
# DzzZ9cN5JWwm6I1ok87xd4k5f6gEBdo0wxTqnwhUq//EfpZsK9OU67Rs4EVNLLL3
# OztatcH714l1bZhycvb3Byjz07LQ6xm+FSx4781FoADk+AR2u1fFkL53VJB0ngtP
# TcSqE4+XrwE1K8ubEXjp8vmJBDxO44ISYuu0RAx1QcIPNLiIncgi8RNq2xgvbnit
# xAW06IQIkwf5fYP+aJg05Hflsc6MlGzbA20oBUd+my7wZPvbpAMxEHwa+zwZgNEL
# cLlVX0e+OWTOt9ojVDLjRrIy2NIphskVXYCVrwL7tNEunTh8NeAPHO0bR0icImpV
# gtnyughlA+XxKfNIigkBTKZ58qK2GpmU65co4b59G6F87VaApvQiM5DkhFP8KvrA
# p5eo6rWNes7k4EuhM6sLdqDVaRa3jma/X/ofxKh/p6FIFJENgvy9TZntyeZsNv53
# Q5m4aS18YS/to7BJ/lu+aSSR/5P8V2mSS9kFP22GctOi0MBk0jpCwRoD+9DtmiG4
# P6+mslFU1UzFyh8SjVfGOe1c/+yfJnatZGZn6Kow4NKtt32xakEnbgOKo3TgigmC
# br/j9re8ngspGGiBoZw/bhZZSxQJCZrmrr9gFd2G9TCCBuwwggTUoAMCAQICEDAP
# b6zdZph0fKlGNqd4LbkwDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMw
# EQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UE
# ChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNB
# IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE5MDUwMjAwMDAwMFoXDTM4MDEx
# ODIzNTk1OVowfTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hl
# c3RlcjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVk
# MSUwIwYDVQQDExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIENBMIICIjANBgkq
# hkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAyBsBr9ksfoiZfQGYPyCQvZyAIVSTuc+g
# PlPvs1rAdtYaBKXOR4O168TMSTTL80VlufmnZBYmCfvVMlJ5LsljwhObtoY/AQWS
# Zm8hq9VxEHmH9EYqzcRaydvXXUlNclYP3MnjU5g6Kh78zlhJ07/zObu5pCNCrNAV
# w3+eolzXOPEWsnDTo8Tfs8VyrC4Kd/wNlFK3/B+VcyQ9ASi8Dw1Ps5EBjm6dJ3VV
# 0Rc7NCF7lwGUr3+Az9ERCleEyX9W4L1GnIK+lJ2/tCCwYH64TfUNP9vQ6oWMilZx
# 0S2UTMiMPNMUopy9Jv/TUyDHYGmbWApU9AXn/TGs+ciFF8e4KRmkKS9G493bkV+f
# PzY+DjBnK0a3Na+WvtpMYMyou58NFNQYxDCYdIIhz2JWtSFzEh79qsoIWId3pBXr
# GVX/0DlULSbuRRo6b83XhPDX8CjFT2SDAtT74t7xvAIo9G3aJ4oG0paH3uhrDvBb
# fel2aZMgHEqXLHcZK5OVmJyXnuuOwXhWxkQl3wYSmgYtnwNe/YOiU2fKsfqNoWTJ
# iJJZy6hGwMnypv99V9sSdvqKQSTUG/xypRSi1K1DHKRJi0E5FAMeKfobpSKupcNN
# gtCN2mu32/cYQFdz8HGj+0p9RTbB942C+rnJDVOAffq2OVgy728YUInXT50zvRq1
# naHelUF6p4MCAwEAAaOCAVowggFWMB8GA1UdIwQYMBaAFFN5v1qqK0rPVIDh2JvA
# nfKyA2bLMB0GA1UdDgQWBBQaofhhGSAPw0F3RSiO0TVfBhIEVTAOBgNVHQ8BAf8E
# BAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcDCDAR
# BgNVHSAECjAIMAYGBFUdIAAwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC51
# c2VydHJ1c3QuY29tL1VTRVJUcnVzdFJTQUNlcnRpZmljYXRpb25BdXRob3JpdHku
# Y3JsMHYGCCsGAQUFBwEBBGowaDA/BggrBgEFBQcwAoYzaHR0cDovL2NydC51c2Vy
# dHJ1c3QuY29tL1VTRVJUcnVzdFJTQUFkZFRydXN0Q0EuY3J0MCUGCCsGAQUFBzAB
# hhlodHRwOi8vb2NzcC51c2VydHJ1c3QuY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBt
# VIGlM10W4bVTgZF13wN6MgstJYQRsrDbKn0qBfW8Oyf0WqC5SVmQKWxhy7VQ2+J9
# +Z8A70DDrdPi5Fb5WEHP8ULlEH3/sHQfj8ZcCfkzXuqgHCZYXPO0EQ/V1cPivNVY
# eL9IduFEZ22PsEMQD43k+ThivxMBxYWjTMXMslMwlaTW9JZWCLjNXH8Blr5yUmo7
# Qjd8Fng5k5OUm7Hcsm1BbWfNyW+QPX9FcsEbI9bCVYRm5LPFZgb289ZLXq2jK0KK
# IZL+qG9aJXBigXNjXqC72NzXStM9r4MGOBIdJIct5PwC1j53BLwENrXnd8ucLo0j
# GLmjwkcd8F3WoXNXBWiap8k3ZR2+6rzYQoNDBaWLpgn/0aGUpk6qPQn1BWy30mRa
# 2Coiwkud8TleTN5IPZs0lpoJX47997FSkc4/ifYcobWpdR9xv1tDXWU9UIFuq/DQ
# 0/yysx+2mZYm9Dx5i1xkzM3uJ5rloMAMcofBbk1a0x7q8ETmMm8c6xdOlMN4ZSA7
# D0GqH+mhQZ3+sbigZSo04N6o+TzmwTC7wKBjLPxcFgCo0MR/6hGdHgbGpm0yXbQ4
# CStJB6r97DDa8acvz7f9+tCjhNknnvsBZne5VhDhIG7GrrH5trrINV0zdo7xfCAM
# KneutaIChrop7rRaALGMq+P5CslUXdS5anSevUiumDGCBC0wggQpAgEBMIGSMH0x
# CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNV
# BAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UEAxMc
# U2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQQIRAJA5f5rSSjoT8r2RXwg4qUMw
# DQYJYIZIAWUDBAICBQCgggFrMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAc
# BgkqhkiG9w0BCQUxDxcNMjMwMTEwMjMwMDEwWjA/BgkqhkiG9w0BCQQxMgQwKVYk
# FKtK3B7hmtKWcFxq2dpOcXl7KDjTBZn6XkE+NXSvYmDXa9SfP97zMWPxz9QgMIHt
# BgsqhkiG9w0BCRACDDGB3TCB2jCB1zAWBBSrNAE6rECXMZ8IGvCzGOGD+A94gTCB
# vAQUAtZbleKDcMFXAJX6iPkj3ZN/rY8wgaMwgY6kgYswgYgxCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwG
# A1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3Qg
# UlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5AhAwD2+s3WaYdHypRjaneC25MA0G
# CSqGSIb3DQEBAQUABIICAGT8iArrERaHj71bhLXUm8HcP62QbFfrCad/nErPDueW
# +3qHFCeHfegJZNBwY86ULVNoxGT0r10o2ejxCbaNrhJwjuTRy85k2fjyTudn0Tro
# 35ja1bB1xZ2rgW6aOd4gc2rPaBjrBqWu7ouBw4wZUKPSTl5HhCbCsJb9iQP5/f0+
# 0KtSi2i8zYM4HvasIU/TuS8iv8LYY5zX6ohQvgjh4r9fWkuWy95H8NffLwNoEZID
# LsYzZDZNpCh9OgMS93mm1h7tMOaih1zB3s+yWhQD8jqjJMYJzK84Nulynt9t8V+X
# Ds2Z+6xMYcGibBLTO9hsev+mWpUJ48Uxy5poYy2Dd50Ej59PWT7PzHNxCT1BZHwl
# DDEdKyM4uDz0HA+KID4v0akGR+x5rUbGg81b7IK6U40FcCP/DhD2Lme73lO1Cpjm
# 8ehvbxMRoNoqyWbkDXbkkX+xKB4+nCAd75Xxtmc3trON7fYBCIQOmS6ie+HT4nHn
# jTMCQFAi/VebKb0njifZxCbHP8pIRHRswLup/QoIxCb21irDikoAjF4eNHTEM2xx
# zOiesmhXa28D8ywKrVBjmqABPNdEW/TaE5zFnN2QSM7HLNZJy6IBMEIHCpxsHhhc
# NLDiudOk/BJC52uZGi9L3DK18MCzgzT4/47uzRCjFElQp5/qG8RjKeEWar48Pd0g
# SIG # End signature block
